// Filename: Unio.C
//
// Description:
//   Unio port I/O virtual device driver source
//
// A.Kolegov (NIIATT) 1.09.1996

#include <windows.h>
#include <winioctl.h>
#include <vddsvc.h>
#include "unioctl.h"
#include "uniodef.h"


// disconnected I/O value
#define FLOATING_IO  0xFF

// I/O hook interceptors
VOID UnioVddByteInput( WORD, PBYTE );
VOID UnioVddByteOutput( WORD, BYTE );


// global variables
HANDLE  hVDD;
HANDLE  hFile;              // Unio device file handle
UNIO_PARAMS UnioParams;     // Unio driver parameters
VDD_IO_PORTRANGE PortRange; // Unio driver port range
BOOLEAN IOHook;             // true if we installed a I/O hooked


// The DllEntryPoint for the UnioVdd which handles intialization
// and termination
//
// Parameters:
//   hVdd       - The handle to the VDD
//   Reason     - flag word thatindicates why Dll Entry Point was invoked
//
// Returned:
//   if( dwReason == DLL_PROCESS_ATTACH )
//       TRUE  - Dll Intialization successful
//       FALSE - Dll Intialization failed
//   else
//       always returns TRUE

BOOL DllEntryPoint( HANDLE hDLL, DWORD dwReason, LPVOID Reserved )
{
  BOOLEAN IoctlResult;
  ULONG   IoctlCode;
  ULONG   ReturnedLength;
  VDD_IO_HANDLERS IOHandlers;

  hVDD = hDLL;

  switch( dwReason ) {
    case DLL_PROCESS_ATTACH:
      // open Unio driver
      hFile = CreateFile(
          "\\\\.\\UnioDev",
          GENERIC_READ | GENERIC_WRITE,
          FILE_SHARE_READ,
          NULL,
          OPEN_EXISTING,
          0,
          NULL
      );
      if( hFile == INVALID_HANDLE_VALUE )
          return FALSE;

      // get Unio driver parameters
      IoctlCode = (ULONG)IOCTL_UNIO_GET_PARAMETERS;
      IoctlResult = DeviceIoControl(
          hFile,                       // Handle to device
          IoctlCode,                   // IO Control code for Read
          NULL,                        // Buffer to driver
          0,                           // Length of buffer in bytes
          &UnioParams,                 // Buffer from driver
          sizeof( UNIO_PARAMS ),       // Length of buffer in bytes
          &ReturnedLength,             // Bytes placed in DataBuffer
          NULL                         // NULL means wait till op. completes
      );
      if( !IoctlResult )
          return FALSE;

      // hook I/O
      IOHandlers.inb_handler   = UnioVddByteInput;
      IOHandlers.inw_handler   = NULL;
      IOHandlers.insb_handler  = NULL;
      IOHandlers.insw_handler  = NULL;
      IOHandlers.outb_handler  = UnioVddByteOutput;
      IOHandlers.outw_handler  = NULL;
      IOHandlers.outsb_handler = NULL;
      IOHandlers.outsw_handler = NULL;
      PortRange.First = (WORD)UnioParams.PortBase;
      PortRange.Last  = (WORD)(UnioParams.PortBase + UnioParams.PortCount - 1);

      IOHook = VDDInstallIOHook( hVDD, 1, &PortRange, &IOHandlers);
      break;

    case DLL_PROCESS_DETACH:
      CloseHandle( hFile );
      if( IOHook )
          VDDDeInstallIOHook( hVDD, 1, &PortRange );
      break;
  }

  return TRUE;
}



VOID UnioVddByteInput( WORD Port, PBYTE Buffer ) {
  BOOLEAN IoctlResult;
  DWORD   IoctlCode;
  DWORD   PortNumber;
  DWORD   DataLength;
  DWORD   ReturnedLength;

  IoctlCode  = (DWORD)IOCTL_UNIO_READ_PORT_UCHAR;
  PortNumber = Port - PortRange.First;
  DataLength = sizeof( *Buffer );

  IoctlResult = DeviceIoControl(
      hFile,                        // Handle to device
      IoctlCode,                    // IO Control code for Read
      &PortNumber,                  // Buffer to driver
      sizeof( PortNumber ),         // Length of buffer in bytes
      Buffer,                       // Buffer from driver
      DataLength,                   // Length of buffer in bytes
      &ReturnedLength,              // Bytes placed in DataBuffer
      NULL                          // NULL means wait till op. completes
  );

  if( !IoctlResult || ReturnedLength != DataLength )
      *Buffer = FLOATING_IO;
}



VOID UnioVddByteOutput( WORD Port, BYTE Data ) {
  BOOLEAN IoctlResult;
  DWORD   IoctlCode;
  PORT_IO DataBuffer;
  DWORD   DataLength;
  DWORD   ReturnedLength;

  IoctlCode = (DWORD)IOCTL_UNIO_WRITE_PORT_UCHAR;
  DataBuffer.PortNumber = Port - PortRange.First;
  DataBuffer.CharData   = Data;
  DataLength = sizeof( DataBuffer.PortNumber ) +
               sizeof( DataBuffer.CharData );

  IoctlResult = DeviceIoControl(
      hFile,                        // Handle to device
      IoctlCode,                    // IO Control code for Read
      &DataBuffer,                  // Buffer to driver
      DataLength,                   // Length of buffer in bytes
      NULL,                         // Buffer from driver
      0,                            // Length of buffer in bytes
      &ReturnedLength,              // Bytes placed in DataBuffer
      NULL                          // NULL means wait till op. completes
  );
}

